<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Collection;

use Exception;
use Erlage\Photogram\Settings;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Tools\Fso\Storage;
use Erlage\Photogram\Tools\Fso\ImageUploader;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\Post\PostSaveTable;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveModel;
use Erlage\Photogram\Data\Models\Post\Save\PostSaveBuilder;
use Erlage\Photogram\Data\Models\Collection\CollectionModel;
use Erlage\Photogram\Data\Tables\Collection\CollectionTable;
use Erlage\Photogram\Data\Models\Collection\CollectionBuilder;
use Erlage\Photogram\Data\Dtos\Collection\CollectionDisplayImageDTO;

final class CollectionActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | add action
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $displayTitleFromReq = self::$request -> findKey(
                CollectionTable::DISPLAY_TITLE,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            $coverPostSaveIdFromReq = self::$request -> findKey(
                CollectionTable::EXTRA_COVER_POST_SAVE_ID,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            $selectedPostSaveIdsFromReq = self::$request -> getCollection(
                PostSaveTable::ID,
                RequestTable::PAYLOAD,
                PostSaveTable::TABLE_NAME
            );

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | build collection
            |--------------------------------------------------------------------------
            */

            $newCollectionModel = (new CollectionBuilder())
                -> setOwnerUserId(self::$authedUserModel -> getId())
                -> setDisplayTitle($displayTitleFromReq)
                -> dispense();

            /*
            |--------------------------------------------------------------------------
            | if display image is provided
            |--------------------------------------------------------------------------
            */

            if (self::isAvailable($coverPostSaveIdFromReq))
            {
                /*
                |--------------------------------------------------------------------------
                | ensure post save exists
                |--------------------------------------------------------------------------
                */

                $postSaveForCoverModel = PostSaveModel::findFromId_throwException($coverPostSaveIdFromReq);

                /*
                |--------------------------------------------------------------------------
                | ensure post exists
                |--------------------------------------------------------------------------
                */

                $targetPostModel = PostModel::findFromId_throwException($postSaveForCoverModel -> getSavedPostId());

                /*
                |--------------------------------------------------------------------------
                | get first image from post
                |--------------------------------------------------------------------------
                */

                $postDisplayImage = $targetPostModel
                    -> getDisplayContent()
                    -> getFirstItem()
                    -> getDisplayItemDTO();

                /*
                |--------------------------------------------------------------------------
                | get raw contents
                |--------------------------------------------------------------------------
                */

                $postImageRawContents = Storage::disk($postDisplayImage -> getHost())
                    -> setObjectId($postDisplayImage -> getOriginalObjectId())
                    -> read();

                /*
                |--------------------------------------------------------------------------
                | create image uploader
                |--------------------------------------------------------------------------
                */

                $host = Settings::getString(ServerConstants::SS_ENUM_STORAGE_DISK_COLLECTION_IMAGES);
                $resolution = Settings::getInt(ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_RES);
                $compressedQuality = Settings::getInt(ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_QUALITY);

                $imageUploader = (new ImageUploader())
                    -> setHost($host)
                    -> setUserId(self::$authedUserModel -> getId())
                    -> setFilespace(DisplayItemDTO::FILESPACE_COLLECTION)
                    -> setSaveResolution($resolution)
                    -> setCompressedQuality($compressedQuality)
                    -> setRawContents($postImageRawContents)
                    -> setExtensionFromIdentifier($postDisplayImage -> getIdentifier());

                /*
                |--------------------------------------------------------------------------
                | finish save
                |--------------------------------------------------------------------------
                */

                $uploadHandle = $imageUploader -> process();

                // check if image upload has succeeded

                if ( ! $uploadHandle -> processed)
                {
                    // upload failed, log internally

                    SystemLogger::internalException(new Exception($uploadHandle -> error));

                    // something went wrong

                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }

                // else get DTO object

                $uploadedImageDTO = $imageUploader -> getDisplayItemDTO();

                /*
                |--------------------------------------------------------------------------
                | update model
                |--------------------------------------------------------------------------
                */

                $newCollectionModel -> update(
                    array(
                        CollectionTable::DISPLAY_IMAGE => (new CollectionDisplayImageDTO())
                            -> setHost($uploadedImageDTO -> getHost())
                            -> setIdentifier($uploadedImageDTO -> getIdentifier()),
                    )
                );
            }

            /*
            |--------------------------------------------------------------------------
            | save
            |--------------------------------------------------------------------------
            */

            $newCollectionModel -> save();

            /*
            |--------------------------------------------------------------------------
            | if user has selected posts, then add them to newly created collection
            |--------------------------------------------------------------------------
            */

            if (0 != \count($selectedPostSaveIdsFromReq))
            {
                $postIdsToInsert = array();

                /**
                 * @var PostSaveModel[]
                 */
                $postSaveModelsFromQuery = CommonQueries::modelsWithMatchingPredicates(
                    PostSaveTable::getTableName(),
                    array(
                        PostSaveTable::ID               => $selectedPostSaveIdsFromReq,
                        PostSaveTable::SAVED_BY_USER_ID => self::$authedUserModel -> getId(),
                    )
                );

                foreach ($postSaveModelsFromQuery as $postSaveModel)
                {
                    if ( ! \in_array($postSaveModel -> getSavedPostId(), $postIdsToInsert))
                    {
                        $postIdsToInsert[] = $postSaveModel -> getSavedPostId();
                    }
                }

                /*
                |--------------------------------------------------------------------------
                | do inserts
                |--------------------------------------------------------------------------
                */

                foreach ($postIdsToInsert as $postId)
                {
                    (new PostSaveBuilder())
                        -> setSavedByUserId(self::$authedUserModel -> getId())
                        -> setSavedPostId($postId)
                        -> setSavedToCollectionId($newCollectionModel -> getId())
                        -> dispense()
                        -> save();
                }
            }

            /*
            |--------------------------------------------------------------------------
            |  if everything is alright, add it map to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(CollectionTable::getTableName(), $newCollectionModel -> getDataMap());
        });
    }

    /*
    |--------------------------------------------------------------------------
    | remove action
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $collectionIdFromReq = self::$request -> findKey(
                CollectionTable::ID,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $collectionIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | get collection
            |--------------------------------------------------------------------------
            */

            $targetCollectionModel = CollectionModel::findFromId_noException($collectionIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | check collection ownership
            |--------------------------------------------------------------------------
            */

            if ($targetCollectionModel -> isModel())
            {
                if (self::$authedUserModel -> getId() == $targetCollectionModel -> getOwnerUserId())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | object references for delete
                    |--------------------------------------------------------------------------
                    */

                    $obsoluteDisplayImageDTO = $targetCollectionModel -> getDisplayImage();

                    /*
                    |--------------------------------------------------------------------------
                    | delete collection
                    |--------------------------------------------------------------------------
                    */

                    $targetCollectionModel -> delete();

                    /*
                    |--------------------------------------------------------------------------
                    | clean objects from storage
                    |--------------------------------------------------------------------------
                    */

                    Storage::disk($obsoluteDisplayImageDTO -> getHost())
                        -> deleteDTO($obsoluteDisplayImageDTO);
                }
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | update action
    |--------------------------------------------------------------------------
    */

    public static function update(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $collectionIdFromReq = self::$request -> findKey(
                CollectionTable::ID,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            $displayTitleFromReq = self::$request -> findKey(
                CollectionTable::DISPLAY_TITLE,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            $coverPostIdFromReq = self::$request -> findKey(
                CollectionTable::EXTRA_COVER_POST_ID,
                RequestTable::PAYLOAD,
                CollectionTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $collectionIdFromReq);

            $updateDisplayTitle = self::isAvailable($displayTitleFromReq);
            $updateDisplayImage = self::isAvailable($coverPostIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure collection exists
            |--------------------------------------------------------------------------
            */

            $targetCollectionModel = CollectionModel::findFromId_throwException($collectionIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure collection ownership
            |--------------------------------------------------------------------------
            */

            if ($targetCollectionModel -> getOwnerUserId() != self::$authedUserModel -> getId())
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | check display title
            |--------------------------------------------------------------------------
            */

            if ($updateDisplayTitle)
            {
                $targetCollectionModel -> update(array(
                    CollectionTable::DISPLAY_TITLE => $displayTitleFromReq,
                ));
            }

            /*
            |--------------------------------------------------------------------------
            | check display image
            |--------------------------------------------------------------------------
            */

            if ($updateDisplayImage)
            {
                /*
                |--------------------------------------------------------------------------
                | ensure post exists
                |--------------------------------------------------------------------------
                */

                $targetPostModel = PostModel::findFromId_throwException($coverPostIdFromReq);

                /*
                |--------------------------------------------------------------------------
                | privacy checks
                |--------------------------------------------------------------------------
                */

                $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

                if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
                {
                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }

                /*
                |--------------------------------------------------------------------------
                | get first image from post
                |--------------------------------------------------------------------------
                */

                $postDisplayImage = $targetPostModel
                    -> getDisplayContent()
                    -> getFirstItem()
                    -> getDisplayItemDTO();

                /*
                |--------------------------------------------------------------------------
                | get raw contents
                |--------------------------------------------------------------------------
                */

                $diskForPostImages = Settings::getString(ServerConstants::SS_ENUM_STORAGE_DISK_POST_IMAGES);

                $postImageRawContents = Storage::disk($postDisplayImage -> getHost())
                    -> setObjectId($postDisplayImage -> getOriginalObjectId())
                    -> read();

                /*
                |--------------------------------------------------------------------------
                | create image uploader
                |--------------------------------------------------------------------------
                */

                $host = Settings::getString(ServerConstants::SS_ENUM_STORAGE_DISK_COLLECTION_IMAGES);
                $resolution = Settings::getInt(ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_RES);
                $compressedQuality = Settings::getInt(ServerConstants::SS_INT_COMPRESS_COLLECTION_COVER_IMAGE_FILE_QUALITY);

                $imageUploader = (new ImageUploader())
                    -> setHost($host)
                    -> setUserId(self::$authedUserModel -> getId())
                    -> setFilespace(DisplayItemDTO::FILESPACE_COLLECTION)
                    -> setSaveResolution($resolution)
                    -> setCompressedQuality($compressedQuality)
                    -> setRawContents($postImageRawContents)
                    -> setExtensionFromIdentifier($postDisplayImage -> getIdentifier());

                /*
                |--------------------------------------------------------------------------
                | finish save
                |--------------------------------------------------------------------------
                */

                $uploadHandle = $imageUploader -> process();

                // check if image upload has succeeded

                if ( ! $uploadHandle -> processed)
                {
                    // upload failed, log internally

                    SystemLogger::internalException(new Exception($uploadHandle -> error));

                    // something went wrong

                    throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                }

                // else get DTO object

                $uploadedImageDTO = $imageUploader -> getDisplayItemDTO();

                /*
                |--------------------------------------------------------------------------
                | update model
                |--------------------------------------------------------------------------
                */

                $targetCollectionModel -> update(
                    array(
                        CollectionTable::DISPLAY_IMAGE => (new CollectionDisplayImageDTO())
                            -> setHost($uploadedImageDTO -> getHost())
                            -> setIdentifier($uploadedImageDTO -> getIdentifier()),
                    )
                );

                /*
                |--------------------------------------------------------------------------
                | save changes made to collection
                |--------------------------------------------------------------------------
                */

                $targetCollectionModel -> save();

                /*
                |--------------------------------------------------------------------------
                | add updated version to reponse
                |--------------------------------------------------------------------------
                */

                self::addToResponse(CollectionTable::getTableName(), $targetCollectionModel -> getDataMap());
            }
        });
    }
}
